@echo off
setlocal enabledelayedexpansion

:: Set console properties
color 0E
title Image to WebP Converter

echo ================================================================
echo              IMAGE TO WEBP CONVERTER
echo ================================================================
echo.
echo This script converts JPG, JPEG, and PNG images to WebP format
echo using Google's official cwebp utility.
echo.
echo ================================================================
echo.

:: Check if cwebp is installed
set "cwebpPath="

:: Check if in PATH
where cwebp >nul 2>&1
if %errorlevel% equ 0 (
    set "cwebpPath=cwebp"
    echo [OK] cwebp utility found in PATH!
    goto cwebp_found
)

echo [INFO] Searching for cwebp installation...

:: Check common user locations first - including nested folders
for /d %%D in ("%USERPROFILE%\Desktop\libwebp*") do (
    if exist "%%D\bin\cwebp.exe" (
        set "cwebpPath=%%D\bin\cwebp.exe"
        echo [OK] Found cwebp at: !cwebpPath!
        goto cwebp_found
    )
    for /d %%E in ("%%D\libwebp*") do (
        if exist "%%E\bin\cwebp.exe" (
            set "cwebpPath=%%E\bin\cwebp.exe"
            echo [OK] Found cwebp at: !cwebpPath!
            goto cwebp_found
        )
    )
)

for /d %%D in ("%USERPROFILE%\Downloads\libwebp*") do (
    if exist "%%D\bin\cwebp.exe" (
        set "cwebpPath=%%D\bin\cwebp.exe"
        echo [OK] Found cwebp at: !cwebpPath!
        goto cwebp_found
    )
    for /d %%E in ("%%D\libwebp*") do (
        if exist "%%E\bin\cwebp.exe" (
            set "cwebpPath=%%E\bin\cwebp.exe"
            echo [OK] Found cwebp at: !cwebpPath!
            goto cwebp_found
        )
    )
)

:: Check system locations
for /d %%D in ("C:\libwebp*") do (
    if exist "%%D\bin\cwebp.exe" (
        set "cwebpPath=%%D\bin\cwebp.exe"
        echo [OK] Found cwebp at: !cwebpPath!
        goto cwebp_found
    )
    for /d %%E in ("%%D\libwebp*") do (
        if exist "%%E\bin\cwebp.exe" (
            set "cwebpPath=%%E\bin\cwebp.exe"
            echo [OK] Found cwebp at: !cwebpPath!
            goto cwebp_found
        )
    )
)

if exist "%ProgramFiles%\libwebp\bin\cwebp.exe" (
    set "cwebpPath=%ProgramFiles%\libwebp\bin\cwebp.exe"
    echo [OK] Found cwebp at: !cwebpPath!
    goto cwebp_found
)

:: Not found - show installation instructions
echo.
echo [ERROR] cwebp utility not found!
echo.
echo cwebp is part of Google's WebP package.
echo.
echo === INSTALLATION INSTRUCTIONS ===
echo.
echo 1. Visit: https://developers.google.com/speed/webp/download
echo 2. Download "libwebp" for Windows (the .zip file)
echo 3. Extract the ZIP file to your Desktop or Downloads folder
echo 4. Run this script again - it will find cwebp automatically
echo.
echo Common extraction locations:
echo   - Desktop: %USERPROFILE%\Desktop\
echo   - Downloads: %USERPROFILE%\Downloads\
echo   - C Drive: C:\
echo.
pause
exit /b 1

:cwebp_found
echo.

:: Display example directory format
echo ================================================================
echo                   DIRECTORY INPUT HELP
echo ================================================================
echo.
echo Please provide the full path to the directory containing images.
echo.
echo EXAMPLES:
echo   C:\Users\YourName\Pictures\Photos
echo   D:\Images\Vacation2024
echo   .\MyImages          (relative path from current location)
echo   .                   (current directory)
echo.
echo TIP: You can drag and drop a folder into this window!
echo.
echo ================================================================
echo.

:: Prompt for directory
set /p "imageDir=Enter the directory path: "

:: Remove quotes if user dragged and dropped
set "imageDir=!imageDir:"=!"

:: Check if directory exists
if not exist "!imageDir!" (
    echo.
    echo [ERROR] Directory not found: !imageDir!
    echo Please check the path and try again.
    echo.
    pause
    exit /b 1
)

:: Check if it's actually a directory
if not exist "!imageDir!\*" (
    echo.
    echo [ERROR] The path exists but is not a directory!
    echo.
    pause
    exit /b 1
)

echo.
echo [OK] Directory found: !imageDir!
echo.

:: Ask for quality setting
echo ================================================================
echo                   QUALITY SETTINGS
echo ================================================================
echo.
echo Choose WebP quality (0-100, higher = better quality, larger file):
echo   Recommended: 80-90 for photos, 90-100 for graphics
echo.
set /p "quality=Enter quality (default 85): "

:: Set default quality if empty
if "!quality!"=="" set "quality=85"

echo.
echo Using quality: !quality!
echo.

:: Ask to create output subdirectory
echo ================================================================
echo                   OUTPUT OPTIONS
echo ================================================================
echo.
echo Choose how to handle the converted WebP files:
echo.
echo   1 = Convert in same directory and DELETE originals after conversion
echo   2 = Save in 'webp_output' subdirectory (keep originals in main directory)
echo.
set /p "outputOption=Choose (1 or 2): "

if "!outputOption!"=="" set "outputOption=2"

if "!outputOption!"=="1" goto option1
if "!outputOption!"=="2" goto option2
goto optiondefault

:option1
set "outputDir=!imageDir!"
set "deleteOriginals=Y"
echo.
echo [OK] Output directory: Same as input
echo [WARNING] Original files will be DELETED after conversion
goto optionend

:option2
set "outputDir=!imageDir!\webp_output"
set "deleteOriginals=N"
if not exist "!outputDir!" mkdir "!outputDir!"
echo.
echo [OK] Output directory: !outputDir!
echo [OK] Original files will be kept
goto optionend

:optiondefault
echo.
echo [ERROR] Invalid option. Using default (Option 2).
set "outputDir=!imageDir!\webp_output"
set "deleteOriginals=N"
if not exist "!outputDir!" mkdir "!outputDir!"
echo [OK] Output directory: !outputDir!

:optionend

echo.
echo ================================================================
echo                   CONVERTING IMAGES
echo ================================================================
echo.
echo Checking and applying EXIF orientation...
echo.

:: Initialize counters
set /a totalConverted=0
set /a totalFailed=0
set /a totalSkipped=0

:: Convert JPG files
for %%F in ("!imageDir!\*.jpg" "!imageDir!\*.jpeg") do (
    if exist "%%F" (
        set "inputFile=%%F"
        set "filename=%%~nF"
        set "outputFile=!outputDir!\!filename!.webp"
        
        echo Converting: %%~nxF
        
        :: Use PowerShell to handle EXIF orientation and convert
        powershell -NoProfile -ExecutionPolicy Bypass -Command "$imgPath='%%F'; $outPath='!outputFile!'; $quality=!quality!; try { Add-Type -AssemblyName System.Drawing; $img=[System.Drawing.Image]::FromFile($imgPath); foreach($prop in $img.PropertyIdList) { if($prop -eq 0x0112) { $orient=$img.GetPropertyItem(0x0112).Value[0]; switch($orient) { 3 {$img.RotateFlip([System.Drawing.RotateFlipType]::Rotate180FlipNone)} 6 {$img.RotateFlip([System.Drawing.RotateFlipType]::Rotate90FlipNone)} 8 {$img.RotateFlip([System.Drawing.RotateFlipType]::Rotate270FlipNone)} }; $img.RemovePropertyItem(0x0112); break; } }; $tempPath=[System.IO.Path]::GetTempFileName()+'.jpg'; $img.Save($tempPath, [System.Drawing.Imaging.ImageFormat]::Jpeg); $img.Dispose(); & '!cwebpPath!' -q $quality $tempPath -o $outPath 2>&1 | Out-Null; Remove-Item $tempPath -Force; exit 0; } catch { $img.Dispose(); & '!cwebpPath!' -q $quality $imgPath -o $outPath 2>&1 | Out-Null; exit 0; }" >nul 2>&1
        
        if !errorlevel! equ 0 (
            echo   [SUCCESS] Created: !filename!.webp
            set /a totalConverted+=1
        ) else (
            echo   [FAILED] Could not convert: %%~nxF
            set /a totalFailed+=1
        )
        echo.
    )
)

:: Convert PNG files (PNG rarely has EXIF orientation issues)
for %%F in ("!imageDir!\*.png") do (
    if exist "%%F" (
        set "inputFile=%%F"
        set "filename=%%~nF"
        set "outputFile=!outputDir!\!filename!.webp"
        
        echo Converting: %%~nxF
        
        "!cwebpPath!" -q !quality! "!inputFile!" -o "!outputFile!" >nul 2>&1
        
        if !errorlevel! equ 0 (
            echo   [SUCCESS] Created: !filename!.webp
            set /a totalConverted+=1
        ) else (
            echo   [FAILED] Could not convert: %%~nxF
            set /a totalFailed+=1
        )
        echo.
    )
)

:: Display summary
echo ================================================================
echo                   CONVERSION COMPLETE
echo ================================================================
echo.
echo Summary:
echo   - Successfully converted: !totalConverted! files
echo   - Failed conversions:     !totalFailed! files
echo   - Output location:        !outputDir!
echo.

:: Delete original files if option 1 was selected
if "!deleteOriginals!"=="Y" (
    if !totalConverted! gtr 0 (
        echo ================================================================
        echo                   DELETING ORIGINAL FILES
        echo ================================================================
        echo.
        
        set /a deletedCount=0
        
        :: Delete JPG files
        for %%F in ("!imageDir!\*.jpg" "!imageDir!\*.jpeg") do (
            if exist "%%F" (
                del "%%F" >nul 2>&1
                if !errorlevel! equ 0 (
                    echo   [DELETED] %%~nxF
                    set /a deletedCount+=1
                ) else (
                    echo   [FAILED] Could not delete: %%~nxF
                )
            )
        )
        
        :: Delete PNG files
        for %%F in ("!imageDir!\*.png") do (
            if exist "%%F" (
                del "%%F" >nul 2>&1
                if !errorlevel! equ 0 (
                    echo   [DELETED] %%~nxF
                    set /a deletedCount+=1
                ) else (
                    echo   [FAILED] Could not delete: %%~nxF
                )
            )
        )
        
        echo.
        echo [OK] Deleted !deletedCount! original files
        echo.
    )
)

if !totalConverted! equ 0 (
    echo [WARNING] No images were converted!
    echo.
    echo Possible reasons:
    echo   - No JPG, JPEG, or PNG files found in the directory
    echo   - Files may be corrupted
    echo   - Insufficient permissions
    echo.
)

echo ================================================================
echo.
echo Press any key to close...
pause >nul
